function [COUNTS,R] = read_align_pos_file( fname, WINDOWS )
%------------------------------------------------------------------------%
%  FILE: read_align_pos_file.m                                           %
%------------------------------------------------------------------------%
%                  SOFTWARE COPYRIGHT NOTICE AGREEMENT                   %
%      This software and its documentation are copyright (2008) by the   %
%  Broad Institute/Massachusetts Institute of Technology.  All rights    %
%  are reserved.  This software is supplied without any warranty or      %
%  guaranteed support whatsoever. Neither the Broad Institute nor MIT    %
%  can be responsible for its use, misuse, or functionality.             %
%------------------------------------------------------------------------%
%  INPUT:  Text file with coordinates of aligned reads                   %
%          Column 1 - Chr, Column 2 - Alignment start, Column 3 - Strand %
%------------------------------------------------------------------------%
%  PARAMETER LIST                                                        %
%    fname           Filename of text file with aligned read coordinates %
%    WINDOWS         Start coordinates for windows of equal size in the  %
%                      alignable portion of the reference genome         %
%      .chr          Chromosome                                          %
%      .breaks       Start coordinates of windows                        %
%------------------------------------------------------------------------%
%  OUTPUT VARIABLES                                                      %
%    COUNTS          Matlab structure with counts of aligned reads       %
%      .chr          Chromosome                                          %
%      .breaks       Start coordinates of windows                        %
%                      NOTE: length(COUNTS.breaks) = WINDOWS.breaks - 23 %
%                      because last bin of histc equals 0                %
%      .counts       Counts of aligned reads                             %
%                                                                        %
%    READS           Matlab structure with aligned read coordinates      %
%      .chr          Chromosome                                          %
%      .pos          Start positions of aligned reads                    %
%------------------------------------------------------------------------%

numChr = length(unique(WINDOWS.chr));

% Load coordinates of genome .marks.visual files generated by ARACHNE
% Column 1 - Chr, Column 2 - Alignment start, Column 3 - Strand
fid = fopen(char(fname));
if fid < 0
    msg = [ 'ERROR: File ' char(fname) ' does not exist'];
    error('read_align_pos_file:fileNotFound',msg );
end

C = textscan(fid,'%u8%f64%s');
fclose(fid);

% Assign contents to variables 
chrUnsorted = C{1};
pos = C{2};
sortpos = zeros(length(pos),1);
sortchr = zeros(length(pos),1,'single');
clear C;

% Count number of reads in genome windows
chrBreaks = ones(numChr,1);
currChrStart = 1;

for c=1:numChr
    currChrPos = sort( pos( find(chrUnsorted==c) ) );
    currChrEnd = currChrStart + length(currChrPos) - 1;
    currWindows = WINDOWS.breaks(find(WINDOWS.chr==c));
    currCounts = histc(currChrPos, currWindows );
    currCounts = currCounts(1:(end-1));    % Last bin of histc are exact matches to final edge
    currChr = repmat(c,length(currCounts),1);
    currPos = currWindows(1:length(currCounts));
    
    if c == 1
	counts = currCounts;
        countchr = currChr;
        countpos = currPos;
    
    else
        counts = [ counts; currCounts ];
        countchr = [ countchr; currChr ];
        countpos = [ countpos; currPos ];
    end
    sortpos(currChrStart:currChrEnd) = currChrPos;
    sortchr(currChrStart:currChrEnd) = c;
    currChrStart = currChrStart + length(currChrPos);  % Update current start position
end

COUNTS.counts = counts;
COUNTS.chr = countchr;
COUNTS.breaks = countpos;

R.chr = sortchr;
R.pos = sortpos;
