function SegSeq( varargin )
%------------------------------------------------------------------------%
%  FILE: SegSeq.m                                                        %
%------------------------------------------------------------------------%
%                  SOFTWARE COPYRIGHT NOTICE AGREEMENT                   %
%      This software and its documentation are copyright (2008) by the   %
%  Broad Institute/Massachusetts Institute of Technology.  All rights    %
%  are reserved.  This software is supplied without any warranty or      %
%  guaranteed support whatsoever. Neither the Broad Institute nor MIT    %
%  can be responsible for its use, misuse, or functionality.             %
%------------------------------------------------------------------------%
%  PRIMARY USAGE (from aligned read coordinates)                         %
%  gp_segment_paired_reads -i sampleinfoFile -s name [OPTIONS]           %
%------------------------------------------------------------------------%
%  ALTERNATIVE USAGE (from Matlab structures                             %
%  gp_segment_paired_reads -n normalReadFile -t tumorReadFile            %
%    -s tumorName [OPTIONS]                                              %
%------------------------------------------------------------------------%
%  PARAMETER LIST                                                        %
%    -t tumorReadFile    Matlab file with coordinates of aligned sequence%
%                        reads generated by <map_solexa_aligned_reads.m> %
%    -W W                Size of local windows (i.e., # of consecutive   %
%					        normal reads)            %
%    -c chrLengthFile    Tab-delimited text file with chromosome lengths %
%                        Col 1) Chr, Col 2) Length (physical coordinates)%
%                        Col 3) Length (alignable genome coordinates)    %
%                        (Default = 'chromInfo_hg18.txt')                %
%    -i sampleinfoFile   Sample information file                         %
%    -s tumorName        Unique name for tumor sample                    %
%    -a initialFP        Number of false positives for initialization of %
%                        candidate breakpoints in normal replicate lanes %
%                        (Default = 1000)                                %
%    -b finalFP          Number of false positives for merged segments   %
%                        in normal replicate lanes  (Default = 10 )      %
%    -d alignableDir Directory of Matlab files with alignable coordinates%
%                      (e.g., ALIGNABLE_hg18_N36_D2_chr1.mat)            %
%                      Note that file names must include directory name  %
%                      in the format ALIGNABLE_[dirName]_chr1.mat        %
%    -e windowFile   Matlab structure with start coordinates of alignable%
%                    windows, e.g. HG18_N36_D2_WINDOWS_100K.mat was      %
%                    calculated with ARACHNE parameters N=36, D=2        %
%    -r calcRatios   Flag to calculate ratios in 100 kb alignable windows%
%                      requires valid values for -d and -e (default=0)   %
%    -m minNormalCount  Minimum number of normal reads in above windows  %
%                       (default=1)                                      %
%    -f medianCenter Flag to center copy-number ratios to 1 (default=0)  %
%    -g germlineCNV  File of germline copy-number variants               %
%                   (default='CNV.verified_080606.combined_reformat.txt')%
%    -v removeCNV    Flag to ignore copy-number variants  (default=1)    %
%------------------------------------------------------------------------%
%  OUTPUT VARIABLES                                                      %
%    matfile         Output Matlab file                                  %
%    segfile         Output tab-delimited text file with one row per     %
%                      copy-number segment
%------------------------------------------------------------------------%
%
%  Date: 2008-11-08
%  LastChangedBy: dchiang


format compact;

%---  PARAMETERS  ---%
%  Process parameters
ARGS = handle_args({'n','t','W','c','i','s','a','b','d','e','r','m','f','g','v'},varargin);
matfileT = ARGS.t;
chrLengthFile = ARGS.c;
tumorName = ARGS.s;

%%-- Use parameter DEFAULTS --%
if isempty(ARGS.W)
    W = 400;
else
    W = str2num(ARGS.W)
end

if isempty(ARGS.c)
    chrLengthFile = 'chromInfo_hg18.txt';
end

if isempty(ARGS.a)
    initFP = 1000;
else
    initFP = str2num(ARGS.a);
end

if isempty(ARGS.b)
    finalFP = 10;
else
    finalFP = str2num(ARGS.b)
end

if isempty(ARGS.d)
    ARGS.d = 'hg18_N36_D2';
end

if isempty(ARGS.r)
    calcRatios = 0;
else
    calcRatios = str2num(ARGS.r);
end

if isempty(ARGS.e) & calcRatios
    ARGS.e = 'HG18_N36_D2_WINDOWS_100K.mat';
end

if isempty(ARGS.s)
   fprintf(1,'USAGE\n');
    error('ERROR: Must specify a sample name [-s]');
end

if isempty(ARGS.m)
    minNormalCount = 1;
else
    minNormalCount = str2num(ARGS.m);
end

if isempty(ARGS.f)
    medianCenter = 0;
else
    medianCenter = 1;
end

if isempty(ARGS.g)
    ARGS.g = 'CNV.verified_080606.combined_reformat.txt';
end

if isempty(ARGS.v)
    removeCNV = 1;
else
    removeCNV = str2num(ARGS.v);
end


%%---  PROCESS ALIGNED READS  ---%
if isempty(ARGS.t)
    if isempty(ARGS.i)
        error('ERROR: Must specify a sample information file [-i]');
    end
    [READN,READT,WINDOWN,WINDOWT,RATIOS] = map_qltout_basic_reads( ARGS.i, ARGS.s, ARGS.d, ARGS.e, minNormalCount, medianCenter, ARGS.g, removeCNV, chrLengthFile, calcRatios );
    T.READN = READN;
    T.READT = READT;

    clear READN;
    clear READT;
% Skip above step if Matlab structure file for aligned reads is specified
else
   try
      T = load(matfileT);

      % Save memory
      clear T.READT.lane;
      clear T.READT.id;
      clear T.READN.id;
   catch
      error(['ERROR: Cannot find ' matfileT]);
   end

end

%%---  ERROR CHECKING  ---%
fid = fopen(chrLengthFile);
if fid == -1
    error(['ERROR: Cannot find chromosome information file ' chrLengthFile]);
end
fclose(fid);




%---  Create or load normal replicates  ---%
if isempty(ARGS.n)
    normalLanes = unique(T.READN.lane);
    midIdx = floor(length(normalLanes)/2);

    if midIdx < 1
        error('ERROR: Two or more replicate lanes from normal sample are required to set p-value cutoffs \n');
    end

    idx1 = find(T.READN.lane>=1 & T.READN.lane<=midIdx);
    idx2 = find(T.READN.lane>midIdx & T.READN.lane<=2*midIdx);

    N.READN.chr = T.READN.chr(idx1);
    N.READN.pos = T.READN.pos(idx1);
    N.READN.lane = T.READN.lane(idx1);
    if calcRatios
       N.READN.alignable = T.READN.alignable(idx1);
    end

    N.READT.chr = T.READN.chr(idx2);
    N.READT.pos = T.READN.pos(idx2);
    N.READT.lane = T.READN.lane(idx2);
    if calcRatios
        N.READT.alignable = T.READN.alignable(idx2);
    end
else
    matfileN = ARGS.n;
    N = load(matfileN);
end

%%---  SEGMENTATION  ---%%
chrs=unique(T.READN.chr);

%---  Find p_bkp from NORMAL  ---%
aN=length(N.READN.chr);      % Total number of reads aligning in normal
aT=length(N.READT.chr);      % Total number of reads aligning in tumor

%---  USE normals to estimate p-values for genome-wide significance  ---%
[BKP,POS,R,pval,p_bkp,p_merge]=segment_solexa_logratios_normals( N.READN, N.READT, chrLengthFile, chrs, W, aN, aT, initFP, finalFP, 1 );

fprintf(1,'p_bkp\t%.2e\n',p_bkp);
fprintf(1,'p_merge\t%.2e\n',p_merge);
aN=length(T.READN.chr);      % Total number of reads aligning in normal
aT=length(T.READT.chr);      % Total number of reads aligning in tumor

%---  Use above p-value cutoffs to segment tumor reads   ---%
[SEG1,SEG,BKP,POS,R,pval]=segment_solexa_logratios( T.READN, T.READT, chrLengthFile, chrs, W, aN, aT, p_bkp, p_merge, 1 );

%---  Save output to disk  ---%
outfile = [ tumorName '_seg_W_' num2str(W) '_initFP_' num2str(initFP) '_finalFP_' num2str(finalFP) '.mat' ]
segfile = [ tumorName '_seg_W_' num2str(W) '_initFP_' num2str(initFP) '_finalFP_' num2str(finalFP) '.txt' ];

save( outfile, 'SEG1', 'SEG', 'BKP', 'POS', 'R', 'T','p_bkp','p_merge','aN','aT','W','-v7.3');
write_segments_logratio( segfile, SEG );

